/*
 * Copyright (c) 1993-2012 David Gay and Gustav Hllberg
 * All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose, without fee, and without written agreement is hereby granted,
 * provided that the above copyright notice and the following two paragraphs
 * appear in all copies of this software.
 *
 * IN NO EVENT SHALL DAVID GAY OR GUSTAV HALLBERG BE LIABLE TO ANY PARTY FOR
 * DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
 * OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF DAVID GAY OR
 * GUSTAV HALLBERG HAVE BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * DAVID GAY AND GUSTAV HALLBERG SPECIFICALLY DISCLAIM ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS ON AN
 * "AS IS" BASIS, AND DAVID GAY AND GUSTAV HALLBERG HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */

#ifndef GLOBAL_H
#define GLOBAL_H

#include "objenv.h"

/* if you change this, make sure to fix the lexer too */
#define GLOBAL_ENV_PREFIX ":"
#define USER_ENV_PREFIX "$"
#define MAX_VARIABLE_LENGTH 1024 /* max characters in a variable name */

extern struct env *environment;
extern struct vector *env_values;
extern struct vector *mvars;
extern struct table *global;
extern struct vector *global_names;

#define GLOBAL_LOOKUP(name, mname)                              \
  static ulong name;                                            \
  if (!name)                                                    \
    {                                                           \
      STATIC_STRING(sstr_ ## name, mname);                      \
      name = mglobal_lookup(GET_STATIC_STRING(sstr_ ## name));  \
    }                                                           \
  ((void)0)

ulong global_lookup(const char *name);
/* Returns: the index for global variable name in environment.
     If name doesn't exist yet, it is created with a variable
     whose value is NULL.
   Modifies: environment
*/

/* true if 'name' is a known global */
bool global_exists(const char *name, ulong *ofs);

ulong mglobal_lookup(struct string *name);
/* Returns: the index for global variable name in environment.
     If name doesn't exist yet, it is created with a variable
     whose value is NULL.
   Modifies: environment
*/

void global_init(void);
/* Effects: Initialises the global environment before use.
*/

#define GVAR(offset) (env_values->data[(offset)])
/* Returns: The global value at 'offset'
*/

#define GCONSTANT(offset) (!integerp(mvars->data[offset]))
/* Returns: a true value if global variable offset is not modifiable
     (i.e., is a 'define' of some module or system-write)
*/

#define GMUTABLE(offset) (mvars->data[offset] == makeint(var_system_mutable))

#define GNAME(offset) ((struct string *)global_names->data[offset])
/* Returns: the name of the global at 'offset'
*/

void check_global_read(ulong goffset);
void check_global_write(ulong goffset, value val);
/* Errors out if the calling code is not allowed to access that global.
   N.b., adjust x64builtins.S if you change this declaration.
*/

static ALWAYS_INLINE long nglobals(void)
{
  return intval(environment->used);
}

#endif
