# Copyright (c) 1993-2015 David Gay and Gustav Hllberg
# All rights reserved.
#
# Permission to use, copy, modify, and distribute this software for any
# purpose, without fee, and without written agreement is hereby granted,
# provided that the above copyright notice and the following two paragraphs
# appear in all copies of this software.
#
# IN NO EVENT SHALL DAVID GAY OR GUSTAV HALLBERG BE LIABLE TO ANY PARTY FOR
# DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
# OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF DAVID GAY OR
# GUSTAV HALLBERG HAVE BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
# DAVID GAY AND GUSTAV HALLBERG SPECIFICALLY DISCLAIM ANY WARRANTIES,
# INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
# FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS ON AN
# "AS IS" BASIS, AND DAVID GAY AND GUSTAV HALLBERG HAVE NO OBLIGATION TO
# PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.

use strict;
use warnings;
use File::Basename;

use Getopt::Long qw(:config bundling no_auto_abbrev no_ignore_case);

sub usage {
    print "Synopsis:\n";
    print "  $0 [options] <header-files>\n";
    print "Options:\n";
    print "  --defs,-d            Generate FOR_DEFS() output\n";
    print "  --help,-h            Show this help\n";
    print "  --output,-o <file>   Send output to <file>\n";
}

my ($defs, $help, $ofilename);
if (!Getopt::Long::GetOptions(
         "defs|d"     => \$defs,
         "help|h"     => \$help,
         "output|o=s" => \$ofilename)) {
    usage();
    exit(1);
}

if (defined($help)) {
    usage();
    exit(0);
}

$#ARGV >= 0 or die "No input files specified";

sub fcmp {
    my $r = dirname($a) cmp dirname($b);
    return $r if $r;
    return basename($a) cmp basename($b);
}

my @files = sort fcmp @ARGV;

undef $/;

my ($ofile, $guarded);

if ($ofilename) {
    open($ofile, '>', $ofilename)
        or die "Failed to open output file: $!";
    $guarded = $ofilename;
} else {
    open($ofile, '>-') or die;
}
print $ofile "/* automatically generated by ", $0, " */\n";

unless (defined($defs)) {
    print $ofile '
#  include "mvalues.h"

#  include "runtime/runtime.h"

';

    for my $file (@files) {
        print $ofile "#include \"$file\"\n";
    }
}

sub emit_def {
    my ($prefix, $name, $subst) = @_;
    die unless $subst =~ m!\As/(.*)/(.*)/\z!;
    my ($from, $to) = ($1, $2);
    my $sname = $name;
    $sname =~ s{$from}{$to}e;
    printf $ofile ("%s  op(%s, \"%s\")", $prefix, $name, $sname);
}

print $ofile "\n";
print $ofile "#define FOR_DEFS(op, sep) \\\n";
my $prefix = "";
for my $filename (@files) {
    open(my $ifile, "<$filename") or die "Failed opening $filename: $!";
    print $ofile ($prefix, "  /* $filename */ \\\n");
    $prefix = "";
    while (<$ifile>) {
        my $text = "";

        my $collect = 1;
        while (m! (?:\A|\G) (.*?) (?:
                  (?: (?: ( end ) | ( start ) ) [ ] mudlle [ ] const )
                      | mudlle [ ] subst [ ] (s/.*?/.*?/)
                      | \z ) !gmxs) {
            $text = $text . $1 if $collect;
            $collect = 0 if $2;
            $collect = 1 if $3;
            $text = $text . "*/\n%%%PREFIX%%% s///\n/*" if $3;
            $text = $text . "*/\n%%%PREFIX%%% $4\n/*" if $4;
        }

        pos = 0;
        $text =~ s!/\*.*?\*/!!gs;

        my $subst = "s///";
        pos = 0;
        while ($text =~ m! %%%PREFIX%%% [ ] (s/.*?/.*?/)
                           | ^\#[ \t]* define [ \t]+ (\w+) [ \t]+ (\S+)
                           | \b enum \s+ (\w+\s+)? { ([^}]*) } !gmx) {
            if ($1) {
                $subst = $1;
            } elsif ($2) {
                next if $3 =~ m/^"/;
                emit_def($prefix, $2, $subst);
                $prefix = " sep() \\\n";
            } else {
                my $enum = $5;
                $enum =~ s!/\*.*?\*/!!gm;
                die unless $enum;
                while ($enum =~ m! (\w+) \s* (?: = \s* [^,]* )? \s* (?:,|\z)
                                   | %%%PREFIX%%% [ ] (s/.*?/.*?/) !gx) {
                    if ($1) {
                        emit_def($prefix, $1, $subst);
                        $prefix = " sep() \\\n";
                    } else {
                        $subst = $2;
                    }
                }
            }
	}
    }
    print STDERR "${filename}:1: warning: no constants found\n"
        unless $prefix;
}

print $ofile "\n\n";

unless (defined($defs)) {
    print $ofile '#define SDEF_OP(n, s) STATIC_STRING(sstr_ ## n, s)

FOR_DEFS(SDEF_OP, SEP_SEMI);

static const struct {
  struct string *name;
  value mval;
} mudlle_int_consts[] = {
#define DEF_OP(n, s) { GET_STATIC_STRING(sstr_ ## n), makeint(n) }
FOR_DEFS(DEF_OP, SEP_COMMA)
};

void mudlle_consts_init(void)
{
  for (size_t i = 0; i < VLENGTH(mudlle_int_consts); ++i)
    system_string_define(mudlle_int_consts[i].name, mudlle_int_consts[i].mval);
}
';
}

undef $guarded;

END {
    unlink($guarded) if defined($guarded);
}
