use strict;
use warnings;

use constant MAXN => 60;

print "/* automatically generated by $0 */

#define __VA_NARGS(";
for my $n (1 .. MAXN) {
    print "_$n, ";
    print "\\\n    " if ($n % 10 == 0);
}
print "N, ...) N

#define VA_NARGS(...) __VA_NARGS(__VA_ARGS__";

for (my $n = MAXN; $n >= 1; --$n) {
    print ", ";
    print "\\\n    " if ($n % 10 == 0);
    print "$n";
}
print ")

#define VA_NPAIRS(...) __VA_NARGS(__VA_ARGS__";
for (my $n = MAXN / 2; $n >= 1; --$n) {
    print ", ";
    print "\\\n    " if ($n % 5 == 0);
    print "$n, X";
}

print ")

";

for my $n (1 .. MAXN) {
    my $p = $n - 1;
    print "#define _INC$p $n\n";
}

print "#define _INC(n) _INC ## n
#define INC(n) _INC(n)

";

for my $n (1 .. MAXN) {
    my $p = $n - 1;
    print "#define _DEC$n $p\n";
}

print "#define _DEC(n) _DEC ## n
#define DEC(n) _DEC(n)

#define _FOR_NA1(n, op, cbarg, sep, arg) op(n, cbarg, arg)
";

for my $n (2 .. MAXN) {
    my $p = $n - 1;
    print "#define _FOR_NA$n(n, op, cbarg, sep, arg, ...) \\
  op(n, cbarg, arg) sep() _FOR_NA$p(INC(n), op, cbarg, sep, __VA_ARGS__)
";
}

print "#define __FOR_NARGS(n, op, cbarg, sep, ...) \\
  _FOR_NA ## n(1, op, cbarg, sep, __VA_ARGS__)
#define _FOR_NARGS(n, op, cbarg, sep, ...) \\
  __FOR_NARGS(n, op, cbarg, sep, __VA_ARGS__)
#define FOR_NARGS(op, cbarg, sep, ...) \\
  _FOR_NARGS(VA_NARGS(__VA_ARGS__), op, cbarg, sep, __VA_ARGS__)

";

for my $n (1 .. MAXN) {
    print "#define ARGN$n(";
    for $a (1 .. $n) {
        print "_$a, ";
        print "\\\n    " if ($a % 10 == 0);
    }
    print "...) _$n\n";
}

print "
#define __FORP1(op, arg, sep, a, b) op(a, b, arg)
";

for (my $n = 2; $n <= MAXN / 2; ++$n) {
    my $p = $n - 1;
    print "#define __FORP$n(op, arg, sep, a, b, ...) \\
  op(a, b, arg) sep() __FORP$p(op, arg, sep, __VA_ARGS__)\n";
}

print "
#define __FORP(n, op, arg, sep, ...) __FORP ## n(op, arg, sep, __VA_ARGS__)
#define _FOR_PAIRS(n, op, arg, sep, ...) \\
  __FORP(n, op, arg, sep, __VA_ARGS__)
#define FOR_PAIRS(op, arg, sep, ...) \\
  _FOR_PAIRS(VA_NPAIRS(__VA_ARGS__), op, arg, sep, __VA_ARGS__)
";

my $TS_SHORT    = 0;
my $TS_IS_FLOAT = 1;
my $TS_RANK     = 2;
my $TS_SIGN     = 3;
my $TS_MIN      = 4;
my $TS_MAX      = 5;

my $RANK_INT = 3;

my @typeary = (
    "_Bool"              => [ "B",   0, 0, 0, "0", "1" ],
    "unsigned char"      => [ "UC",  0, 1, 0, "0",         "UCHAR_MAX" ],
    "signed char"        => [ "SC",  0, 1, 1, "SCHAR_MIN", "SCHAR_MAX" ],
    "char"               => [ "C",   0, 1, 2, "CHAR_MIN",  "CHAR_MAX" ],
    "unsigned short"     => [ "US",  0, 2, 0, "0",         "USHRT_MAX" ],
    "short"              => [ "S",   0, 2, 1, "SHRT_MIN",  "SHRT_MAX" ],
    "unsigned"           => [ "U",   0, 3, 0, "0",         "UINT_MAX" ],
    "int"                => [ "I",   0, 3, 1, "INT_MIN",   "INT_MAX" ],
    "unsigned long"      => [ "UL",  0, 4, 0, "0",         "ULONG_MAX" ],
    "long"               => [ "L",   0, 4, 1, "LONG_MIN",  "LONG_MAX" ],
    "unsigned long long" => [ "ULL", 0, 5, 0, "0",         "ULLONG_MAX" ],
    "long long"          => [ "LL",  0, 5, 1, "LLONG_MIN", "LLONG_MAX" ],
    "float"              => [ "F",   1, 6, 1, 0, 0 ],
    "double"             => [ "D",   1, 7, 1, 0, 0 ]
);

my %types;
my %unsigns;
my %signs;
my @typenames;

my $last_unsigned;
for (my $i = 0; $i <= $#typeary; $i += 2) {
    my $tname = $typeary[$i];
    my $tref = $typeary[$i + 1];
    if ($tref->[$TS_SIGN] == 0) {
        $last_unsigned = $tname;
        $signs{$tname} = $typeary[$i + 2] unless $tref->[$TS_RANK] == 0;
    } elsif ($tref->[$TS_SIGN] == 1) {
        $signs{$tname} = $tname;
    }
    $unsigns{$tname} = $last_unsigned;
    $typenames[$#typenames + 1] = $tname;
    $types{$tname} = $tref;
}

sub zero_const {
    my ($t) = @_;
    my $tref = $types{$t};

    return "0.0f" if ($tref->[$TS_SHORT] eq "F");
    return "0.0" if ($tref->[$TS_SHORT] eq "D");

    return "0" if ($tref->[$TS_RANK] <= $RANK_INT);
    return "0" . $tref->[$TS_SHORT];
}

print "
#define MIN_VALUE(v) _Generic( \\
  (v)";
for my $type (@typenames) {
    my $tref = $types{$type};
    next if $tref->[$TS_IS_FLOAT];
    print ", \\\n  ${type}: " . $tref->[$TS_MIN];
}
print ")

#define MAX_VALUE(v) _Generic( \\
  (v)";
for my $type (@typenames) {
    my $tref = $types{$type};
    next if $tref->[$TS_IS_FLOAT];
    print ", \\\n  ${type}: " . $tref->[$TS_MAX];
}
print ")\n";

sub safe_cast {
    my ($tref) = @_;
    return $tref->[$TS_IS_FLOAT] || $tref->[$TS_RANK] == 0;
}

sub need_clamp {
    my ($dtref, $vtref) = @_;
    return 0 if $dtref eq $vtref;
    return 0 if $vtref->[$TS_RANK] == 0;
    return 0 if safe_cast($dtref);

    my $dsign = $dtref->[$TS_SIGN];
    my $vsign = $vtref->[$TS_SIGN];

    return 0 if $vsign == 2 && $dtref->[$TS_RANK] > $vtref->[$TS_RANK];

    return $dsign != $vsign || $dtref->[$TS_RANK] <= $vtref->[$TS_RANK];
}

for my $dtype (@typenames) {
    my $dtref = $types{$dtype};

    my $dsign = $dtref->[$TS_SIGN];

    my $dlcshort = lc $dtref->[$TS_SHORT];

    if (safe_cast $dtref) {
        print "
static ALWAYS_INLINE $dtype clamp_to_${dlcshort}($dtype v)
{
  return v;
}
";
        next;
    }

    for my $vtype (@typenames) {
        my $vtref = $types{$vtype};
        my $vsign = $vtref->[$TS_SIGN];

        next unless need_clamp($dtref, $vtref);

        next if $vtref->[$TS_RANK] < $RANK_INT;
        next if $vtref->[$TS_RANK] < $dtref->[$TS_RANK];

        my $vlcshort = lc $vtref->[$TS_SHORT];

        print "
static ALWAYS_INLINE $dtype clamp_${vlcshort}_to_${dlcshort}($vtype v)
{
  return ";
        my $varname = "v";
        if ($vsign != 0 && $dsign == 0 && !$vtref->[$TS_IS_FLOAT]) {
            print "v < ${\$dtref->[$TS_MIN]} ? ${\$dtref->[$TS_MIN]} : ";
            $varname = "(${unsigns{$vtype}})v";
        }

        my $cast = $vtref->[$TS_IS_FLOAT] ? "(${vtype})" : "";

        if (($vtref->[$TS_IS_FLOAT] && !$dtref->[$TS_IS_FLOAT])
            || ($vsign != 0 && $dsign != 0
                && $dtref->[$TS_RANK] < $vtref->[$TS_RANK])) {
            my $mincast = ($vtref->[$TS_IS_FLOAT] && $dsign != 0) ? $cast : "";
            print "${varname} < ${mincast}${\$dtref->[$TS_MIN]}"
                . " ? ${\$dtref->[$TS_MIN]} : ";
        }

        unless (($vsign == $dsign || $dsign == 0)
                && $dtref->[$TS_RANK] >= $vtref->[$TS_RANK]) {
            print "${varname} <= ${cast}${\$dtref->[$TS_MAX]}"
                . " ? ${varname} : ${\$dtref->[$TS_MAX]}";
        } else {
            print "${varname}";
        }
        print ";
}
";
    }
}

print "\n#define CLAMP_TO(dst, val) _Generic( \\
  (dst)";

for my $dtype (@typenames) {
    my $dtref = $types{$dtype};

    print ", \\\n  ${dtype}: ";

    my $dlcshort = lc $dtref->[$TS_SHORT];

    if (safe_cast($dtref)) {
        print "clamp_to_${dlcshort}";
        next;
    }

    my $dsign = $dtref->[$TS_SIGN];

    my $zero = zero_const($dtref->[$TS_RANK] < $RANK_INT
                          ? "int"
                          : $signs{$dtype});

    print "_Generic( \\
    (val) + ${zero}";
    for my $vtype (@typenames) {
        my $vtref = $types{$vtype};

        next if $vtref->[$TS_RANK] < $RANK_INT;
        next if $vtref->[$TS_RANK] < $dtref->[$TS_RANK];

        print ", \\\n    $vtype: ";
        unless (need_clamp($dtref, $vtref)) {
            print "(${dtype})(val)";
            next;
        }

        my $vsign = $vtref->[$TS_SIGN];

        my @sign_effs = ( "u", "i", "c" );
        my $veff = ($dsign == 2 || $vtref->[$TS_RANK] >= $RANK_INT
                    ? lc($vtref->[$TS_SHORT])
                    : $sign_effs[$vsign]);

        my $dlcshort = lc $dtref->[$TS_SHORT];
        print "clamp_${veff}_to_${dlcshort}";
    }
    print ")";
}
print ")(val)

#define IS_UNSIGNED(val) _Generic( \\
  (val)";
for my $vtype (@typenames) {
    my $vtref = $types{$vtype};

    print ", \\\n  ${vtype}: ";
    if ($vtref->[$TS_SIGN] == 0) {
        print "1";
    } elsif ($vtref->[$TS_SIGN] == 2) {
        print $vtref->[$TS_MIN] . " == 0";
    } else {
        print "0";
    }
}
print ")\n";

sub min_max {
    my ($min) = @_;
    my $MIN = uc $min;

    for my $atype (@typenames) {
        my $atref = $types{$atype};
        my $atsign = $atref->[$TS_SIGN];

        next if $atref->[$TS_IS_FLOAT];

        next if $atref->[$TS_RANK] < $RANK_INT;

        my $alcshort = lc $atref->[$TS_SHORT];

        my $rtype = $min eq "cmp" ? "int" : $atype;

        print "
static ALWAYS_INLINE $rtype ${min}_${alcshort}(${atype} a, ${atype} b)
{
  return a ${$min eq 'max' ? \'>' : \'<'} b ? ";
        if ($min eq "cmp") {
            print "-1 : a > b";
        } else {
            print "a : b";
        }
        print ";\n}\n";

        for my $btype (@typenames) {
            my $btref = $types{$btype};
            my $btsign = $btref->[$TS_SIGN];

            next if $btref->[$TS_RANK] != $atref->[$TS_RANK];
            next if $btsign == $atsign;
            next if $btref->[$TS_IS_FLOAT];

            my $alcshort = lc($atref->[$TS_SHORT]);
            my $blcshort = lc($btref->[$TS_SHORT]);

            my $rtype = ($min eq "cmp"
                         ? "int"
                         : ($min eq "min"
                            ? ($atsign ? $atype : $btype)
                            : ($atref->[$TS_RANK] > $btref->[$TS_RANK]
                               ? $unsigns{$atype} : $unsigns{$btype})));
            print "
static ALWAYS_INLINE ${rtype} ${min}_${alcshort}_${blcshort}(${atype} a,"
                . " ${btype} b)
{
  return ";
            if ($atsign) {
                if ($min eq "cmp") {
                    print "(a < 0 || (${unsigns{$atype}})a < b)"
                        . " ? -1 : (${unsigns{$atype}})a > b";
                } elsif ($min eq "min") {
                    print "(a < 0 || (${unsigns{$atype}})a < b)"
                        . " ? a : ($rtype)b";
                } else {
                    print "(a > 0 && (${unsigns{$atype}})a > b)"
                        . " ? ($rtype)a : b";
                }
            } else {
                if ($min eq "cmp") {
                    print "b < 0 ? 1 : a < (${unsigns{$btype}})b ? -1 :"
                        . " a > (${unsigns{$btype}})b";
                } elsif ($min eq "min") {
                    print "(b < 0 || (${unsigns{$btype}})b < a)"
                        . " ? b : ($rtype)a";
                } else {
                    print "(b > 0 && (${unsigns{$btype}})b > a)"
                        . " ? ($rtype)b : a";
                }
            }
            print ";\n}\n";
        }
    }

    print "
#undef ${MIN}
#define ${MIN}(a, b) _Generic( \\
  (a) + 0";

    for my $atype (@typenames) {
        my $atref = $types{$atype};

        next if $atref->[$TS_RANK] < $RANK_INT;

        my $ashort = $atref->[$TS_SHORT];
        my $alcshort = lc $ashort;

        my $zero = zero_const($signs{$atype});

        print ", \\\n  ${atype}: _Generic( \\
    (b) + ${zero}";

        for my $btype (@typenames) {
            my $btref = $types{$btype};

            next if $btref->[$TS_RANK] < $atref->[$TS_RANK];

            my $bshort = $btref->[$TS_SHORT];
            my $blcshort = lc $bshort;

            my $lhstype = $atype;
            if ($btref->[$TS_RANK] > $atref->[$TS_RANK]) {
                $lhstype = ($atref->[$TS_SIGN]
                            ? ${signs{$btype}}
                            : ${unsigns{$btype}});
            }
            my $lhsref = $types{$lhstype};
            my $lhslcshort = lc $lhsref->[$TS_SHORT];

            print ", \\\n    ${btype}: ";
            if (!$atref->[$TS_IS_FLOAT] && !$btref->[$TS_IS_FLOAT]
                && $atref->[$TS_SIGN] != $btref->[$TS_SIGN]) {
                print "${min}_${lhslcshort}_${blcshort}";
            } elsif ($atref->[$TS_RANK] > $btref->[$TS_RANK]) {
                print "${min}_${lhslcshort}";
            } else {
                print "${min}_${blcshort}";
            }
        }
        print ")";
    }
    print ")((a), (b))\n";
}

print "\
/* defined in utils.c */
int cmp_f(float a, float b);
int cmp_d(double a, double b);

float max_f(float a, float b);
double max_d(double a, double b);

float min_f(float a, float b);
double min_d(double a, double b);
";

min_max "min";
min_max "max";
min_max "cmp";
