/*
 * Copyright (c) 1993-2012 David Gay and Gustav Hllberg
 * All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose, without fee, and without written agreement is hereby granted,
 * provided that the above copyright notice and the following two paragraphs
 * appear in all copies of this software.
 *
 * IN NO EVENT SHALL DAVID GAY OR GUSTAV HALLBERG BE LIABLE TO ANY PARTY FOR
 * DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
 * OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF DAVID GAY OR
 * GUSTAV HALLBERG HAVE BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * DAVID GAY AND GUSTAV HALLBERG SPECIFICALLY DISCLAIM ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS ON AN
 * "AS IS" BASIS, AND DAVID GAY AND GUSTAV HALLBERG HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */

#include "mudlle-config.h"

#include <ctype.h>

#include "alloc.h"
#ifdef ALLOC_STATS
#  include "context.h"
#endif
#include "mvalues.h"
#include "strbuf.h"
#include "types.h"
#include "utils.h"

const char *const mudlle_type_names[] = {
  [type_icode]        = "icode",
  [type_closure]      = "closure",
  [type_variable]     = "variable",
  [type_internal]     = "internal",
  [type_primitive]    = "primitive",
  [type_varargs]      = "varargs",
  [type_secure]       = "secure",
  [type_integer]      = "integer",
  [type_string]       = "string",
  [type_vector]       = "vector",
  [type_pair]         = "pair",
  [type_symbol]       = "symbol",
  [type_table]        = "table",
  [type_private]      = "private",
  [type_object]       = "object",
  [type_character]    = "character",
  [type_gone]         = "gone",
  [type_oport]        = "oport",
  [type_mcode]        = "mcode",
  [type_float]        = "float",
  [type_bigint]       = "bigint",
  [type_null]         = "null",
  [type_connection]   = "connection",
  [type_cookie]       = "cookie",
  [type_file]         = "file",
  [type_weak_ref]     = "weak_ref",
  [type_regexp]       = "regexp",
  [stype_none]        = "none",
  [stype_any]         = "any",
  [stype_function]    = "function",
  [stype_list]        = "list",
  [stype_float_like]  = "float_like",
  [stype_bigint_like] = "bigint_like",
  [stype_false]       = "false"
};
CASSERT_VLEN(mudlle_type_names, 34);

#ifdef ALLOC_STATS

struct file_line
{
  struct file_line *next;
  char *file;
  int line;
  unsigned key;
  int count;
  int size;
};

struct file_line_stats
{
  int size, used;
  struct file_line **slots;
};

static struct file_line_stats type_alloc_stats[mudlle_types];

static unsigned str_int_hash(const char *str, int line)
{
  unsigned key = line;
  while (*str)
    {
      key = (key * 31) ^ *(unsigned char *)str;
      ++str;
    }
  return key;
}

struct vector *get_alloc_stats(void)
{
  int entries = 0;
  for (enum mudlle_type t = 0; t < mudlle_types; ++t)
    entries += type_alloc_stats[t].used;

  struct vector *res = alloc_vector(entries);
  struct vector *vec = NULL;

  int used = 0;

  GCPRO(res, vec);
  for (enum mudlle_type t = 0; t < mudlle_types; ++t)
    for (int i = 0; i < type_alloc_stats[t].size; ++i)
      for (struct file_line *fl = type_alloc_stats[t].slots[i];
           fl; fl = fl->next)
        {
          vec = alloc_vector(5);
          SET_VECTOR(vec, 0, alloc_string(fl->file));
          vec->data[1] = makeint(fl->line);
          vec->data[2] = makeint(t);
          vec->data[3] = makeint(fl->count);
          vec->data[4] = makeint(fl->size);
          fl->count = fl->size = 0;

          res->data[used++] = vec;
        }
  UNGCPRO();

  return res;
}

static struct file_line *file_line_stat_lookup(struct file_line_stats *stats,
					       const char *file,
					       int line)
{
  if (stats->size == 0)
    return NULL;

  unsigned key = str_int_hash(file, line);
  for (struct file_line *fl = stats->slots[key % stats->size];
       fl != NULL;
       fl = fl->next)
    if (fl->key == key && fl->line == line && strcmp(fl->file, file) == 0)
      return fl;

  return NULL;
}

static void inc_file_line(struct file_line_stats *stats, const char *file,
			  int line, int size)
{
  struct file_line *data = file_line_stat_lookup(stats, file, line);

  if (data != NULL)
    {
      ++data->count;
      data->size += size;
      return;
    }

  unsigned key = str_int_hash(file, line);

  if (stats->used * 3 >= stats->size * 2)
    {
      int osize = stats->size;
      stats->size = osize ? osize * 2 : 16;
      struct file_line **new = calloc(stats->size, sizeof *new);

      for (int i = 0; i < osize; ++i)
        {
          for (struct file_line *fl = stats->slots[i], *next;
               fl; fl = next)
            {
              next = fl->next;
              int slot = fl->key % stats->size;
              fl->next = new[slot];
              new[slot] = fl;
            }
        }

      free(stats->slots);
      stats->slots = new;
    }

  data = malloc(sizeof *data);
  *data = (struct file_line){
    .key   = key,
    .file  = strdup(file),
    .line  = line,
    .count = 1,
    .size  = size,
    .next  = stats->slots[key % stats->size],
  };
  stats->slots[key % stats->size] = data;

  ++stats->used;
}

void record_allocation(enum mudlle_type type, long size)
{
  const char *filename = "<nowhere>";
  int line = 0;

  struct call_stack *cs = call_stack;
  if (cs == NULL)
    goto done;

  for (; cs->next; cs = cs->next)
    switch (cs->type)
      {
      case call_c:
      case call_compiled:
      case call_primop:
      case call_invalid:
      case call_invalid_argp:
      case call_invalid_argv:
      case call_session:
        break;
      case call_mstring:
      case call_string_args:
      case call_string_argv:
        goto done;
      case call_bytecode:
        {
          struct call_stack_mudlle *mcs = (struct call_stack_mudlle *)cs;
          filename = mcs->fn->code->filename->str;
          line = mcs->fn->code->lineno;
          goto done;
        }
      }

 done:
  inc_file_line(&type_alloc_stats[type], filename, line, size);
}

#endif /* ALLOC_STATS */

struct closure *alloc_closure_noinit(ulong nb_variables)
{
  ulong fields = nb_variables + grecord_fields(struct closure);
  return make_readonly(
    alloc_record_noinit(garbage_mtemp, type_closure, fields));
}

struct closure *alloc_closure0(struct code *code)
{
  GCPRO(code);

#ifdef ALLOC_STATS
  record_allocation(type_closure, sizeof (struct obj) + sizeof (value));
#endif

  struct closure *newp = (struct closure *)allocate_mtemp(
    type_closure, grecord_fields(*newp));
  newp->code = code;
  assert(immutablep(code));
  newp->o.flags |= OBJ_READONLY | OBJ_IMMUTABLE;
  UNGCPRO();

  return newp;
}

/* allocate uninitialized string for 'len' characters */
struct string *alloc_string_noinit(size_t len)
{
  struct string *result = (struct string *)allocate_string(
    type_string, len + 1);
  result->str[len] = 0;
  return result;
}

struct string *alloc_string(const char *s)
{
  if (s == NULL) s = "(null)";
  return alloc_string_length(s, strlen(s));
}

struct string *alloc_string_length(const char *str, size_t len)
{
  if (str == NULL && len > 0)
    {
      abort();
    }

  struct string *newp = alloc_string_noinit(len);
  if (len > 0)
    memcpy(newp->str, str, len);
  return newp;
}

struct string *mudlle_string_copy(struct string *s)
{
  GCPRO(s);
  size_t len = string_len(s);
  struct string *result = alloc_string_noinit(len);
  memcpy(result->str, s->str, len);
  UNGCPRO();
  return result;
}

char *mudlle_string_dup(struct string *s)
{
  size_t size = string_len(s) + 1;
  char *r = malloc(size);
  memcpy(r, s->str, size);
  return r;
}

struct string *safe_alloc_string(const char *s)
{
  return alloc_string(s ? s : "");
}

struct string *strbuf_to_mudlle(const struct strbuf *sb)
{
  return alloc_string_length(sb_str(sb), sb_len(sb));
}

struct variable *alloc_variable(value val)
{
  GCCHECK(val);
  GCPRO(val);
  struct variable *newp = ALLOC_RECORD_NOINIT(variable, 1);
  newp->vvalue = val;
  UNGCPRO();

  return newp;
}

struct mudlle_float *alloc_float(double d)
{
  struct mudlle_float *newp = (struct mudlle_float *)allocate_string(
    type_float, sizeof d);
  newp->d = d;
  newp->o.flags |= OBJ_READONLY | OBJ_IMMUTABLE;
  return newp;
}

/* always run this before doing any operations on bi, in case a GC has
   moved the data around */
#ifdef USE_GMP
void check_bigint(struct bigint *bi)
{
  bi->mpz[0]._mp_d = &bi->limbs[0];
}

struct bigint *alloc_bigint(mpz_t mpz)
{
  struct bigint *newp = (struct bigint *)allocate_string(
    type_bigint,
    sizeof (mpz_t) + sizeof (mp_limb_t) * mpz[0]._mp_alloc);
  newp->mpz[0]._mp_alloc = mpz[0]._mp_alloc;
  newp->mpz[0]._mp_size = mpz[0]._mp_size;
  newp->mpz[0]._mp_d = (mp_limb_t *)0xdeadbeef;
  memcpy(&newp->limbs[0], mpz->_mp_d,
	 sizeof (mp_limb_t) * ABS(mpz[0]._mp_size));
  newp->o.flags |= OBJ_READONLY | OBJ_IMMUTABLE;
  return newp;
}
#endif

struct symbol *alloc_symbol(struct string *name, value data)
{
  GCPRO(name, data);
  assert(obj_readonlyp(&name->o));
  struct symbol *newp = ALLOC_RECORD_NOINIT(symbol, 2);
  UNGCPRO();
  newp->name = name;
  newp->data = data;
  return newp;
}

struct vector *alloc_vector(ulong size)
{
  return (struct vector *)allocate_record(type_vector, size);
}

struct vector *make_vector(unsigned argc, value args[static argc])
{
  if (argc == 0)
    return ALLOC_RECORD_NOINIT(vector, 0);

  struct gcpro gcpros[argc];
  for (unsigned i = 0; i < argc; ++i)
    GCPROV(gcpros[i], args[i]);
  struct vector *v = ALLOC_RECORD_NOINIT(vector, argc);
  UNGCPROV(gcpros[0]);
  memcpy(v->data, args, argc * sizeof args[0]);
  return v;
}

struct list *alloc_list(value car, value cdr)
{
  GCPRO(car, cdr);
  struct list *newp = (struct list *)alloc_record_noinit(
    garbage_record, type_pair, grecord_fields(*newp));
  UNGCPRO();
  newp->car = car;
  newp->cdr = cdr;
  return newp;
}


/*
 * Converts the string 'sp' of length 'len' into '*l' and returns true. On
 * over/underflow or illegal characters, it returns false.
 * If 'allow_one_overflow', -MIN_TAGGED_INT is accepted (for decimal only).
 * 'base' is the base; or 0 to determined by prefix.
 */
bool mudlle_strtolong(const char *sp, size_t len, long *l, int base,
                      bool allow_one_overflow)
{
  const char *const end = sp + len;

  assert(base == 0 || (base >= 2 && base <= 36));

  while (sp < end && isspace(*(unsigned char *)sp))
    ++sp;

  if (sp == end)
    return false;

  int sign;
  if (*sp == '+')
    {
      sign = 1;
      ++sp;
    }
  else if (*sp == '-')
    {
      sign = -1;
      ++sp;
    }
  else
    sign = 0;

  if (sp == end)
    return false;

  if (*sp == '0')
    {
      unsigned char t = sp[1];
      if (t == 'x' || t == 'X')
        {
          if (base != 0 && base != 16)
            return false;
          base = 16;
          sp += 2;
        }
      else if (t == 'b' || t == 'B')
        {
          if (base != 0 && base != 2)
            return false;
          base = 2;
          sp += 2;
        }
      else
        {
          ++sp;
          if (sp == end)
            {
              /* optimize for common case */
              *l = 0;
              return true;
            }
          if (base == 0)
            base = 8;
          goto skip_empty_check;
        }
    }
  else if (base == 0)
    base = 10;

  if (sp == end)
    return false;

 skip_empty_check:;

  /* only allow the sign bit to be set if no + or - and base != 10 */
  long lim = (!sign && base != 10
              ? MAX_TAGGED_UINT
              : ((sign == -1 || allow_one_overflow)
                 ? -MIN_TAGGED_INT
                 : MAX_TAGGED_INT));

  long limrad = lim / base;
  long n = 0;
  for (;;)
    {
      unsigned char c = *sp++;
      int d = xdigit_val(c);
      if (d < 0)
	return false;

      if (d >= base)
        return false;

      n = n * base + d;
      if (n > lim)
	return false;

      if (sp == end)
	{
	  if (!sign && base != 10)
            n = mudlle_sign_extend(n);
	  *l = sign == -1 ? -n : n;
	  return true;
	}

      if (n > limrad)
	return false;
    }
}

/* returns true on success; sp[len] must be NUL */
bool mudlle_strtofloat(const char *sp, size_t len, double *d)
{
  assert(sp[len] == 0);         /* could make a copy to handle this case */
  const char *endp;
  *d = strtod(sp, (char **)&endp);
  return *sp && endp == sp + len;
}
